// ToolPlugIn.cs
//
// This file implements an IPlugIn-derived class. The
// implementation creates a simple tool inside the user 
// interface of the TLA.
/////////////////////////////////////////////////////////////

using System;
using System.Windows.Forms;
using Tektronix.LogicAnalyzer.PlugIn;
using Tektronix.LogicAnalyzer.TpiNet;

namespace CSharpSamplePlugIn
{
	/// <summary>
	/// CSToolPlugIn is a simple IPlugIn implementation. It 
	/// demonstrates the basics of TLA plug-in development.
	/// </summary>

    // The use of the PlugInIdentity Attribute causes the TLA to create
    // a tool bar button and a Tools menu item to represent the installed
    // plug-in in the user interface. Pressing the toolbar button will
    // activate a single instance of the plug-in and its UI.
    // PlugInInstantiation Attribute is also applied to the clas. The 
    // arguments to PlugInInstantiation cause a single instance of the 
    // plug-in class to instantiated whenever the TLA system is defaulted.
	[PlugInIdentity("CSharp Sample Tool")]
	[PlugInInstantiation(true, true)]
	public class CSToolPlugIn : IPlugIn
	{
		// Data Members:
		private ToolForm form = null;
		private bool isValid = true;
		private bool isGarbage = false;
		
		
		// Every plug-in must have a default constructor
		// (a constructor that takes no arguments.
		public CSToolPlugIn()
		{
		}
		
		
		// The plug-in subscribes to ToolForm.Closed. This
		// event handler is called when the event occurs.
		// The plug-in responds by disposing itself, which
		// deletes the plug-in instance from the system.
		void OnFormClosed(object sender, EventArgs args) {
		    Dispose();
		}

    // IPlugIn Members		
		
		// This method uses the supplied ITlaPlugInSupport object to 
		// embed its .NET form into main window of the TLA application.
		public bool Initialize(ITlaPlugInSupport support, bool userInit)
		{
		    form = new ToolForm(support);
		    support.TopLevelFormToChild(form);
		    
			// This lets the plug-in know when the user 
			// closes the plug-in UI.
			form.Closed += new EventHandler(OnFormClosed);
		    
		    return true;
		}
		
		
	    // This property provides a reference to the primary user inteface
	    // window of the plug-in. The TLA will use this property to find and 
	    // activate the main window of a plug-in when its tool bar button 
	    // is pressed.
		public Form PrimaryForm
		{
		    get {
		        return form;
		    }
		}
		
		
	// IDisposable Member
	
		// This method invalidates the plug-in and disposes of its window.
		public void Dispose()
		{
		    if (!isGarbage) {
		        if (null != form) {
        		    form.Closed -= new EventHandler(OnFormClosed);
		            form.Dispose();
		        }
    		    
    		    isValid = false;
    		    isGarbage = true;
    		    
		        if (null != ValidityChanged) {
		            ValidityChanged(this, EventArgs.Empty);
		        }
		    }
		}
		
	// IValidity Members
		
		// This event is raised whenever either of the two 
		// properties below change.
		public event EventHandler ValidityChanged;

		// Property that indicates whether the plug-in is 
		// valid to be part of the system.
		public bool IsValid
		{
		    get {
		        return isValid;
		    }
		}
		
		// Property that indicates whether the plug-in is permanently
		// invalid. This value should always be set to true when a
		// plug-in is disposed.
		public bool IsGarbage 
		{
		    get {
		        return isGarbage;
		    }
		}
	}
}
